/*
 * avrdude - A Downloader/Uploader for AVR device programmers
 * Copyright (C) 2000-2006  Brian S. Dean <bsd@bsdhome.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// This stuff is all adapted from par.c & usb_libusb.c cause its the only code i know well 
// and usbasp is based on code that is oddly licensed. :(

#include "ac_cfg.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>

#include "avr.h"
#include "pgm.h"
#include "bitbang.h"
#include "usbspi.h"

extern char *progname;

#if defined(HAVE_LIBUSB)      // we use LIBUSB to talk to the board

#define USB_TIMEOUT 500 // ms

#include <usb.h>

#include "serial.h"
#include "usbdevs.h"

extern int verbose;
static usb_dev_handle *usbhandle = 0;
//static char usbbuf[USBDEV_MAX_XFER];
//static int buflen = -1, bufptr;
//static int usb_interface;

static	int usb_out ( int req, int val, int index, char* buf, int buflen, int umax )
{
  int	n;
  int	timeout;
  
  timeout = USB_TIMEOUT + (buflen * umax) / 1000;
  n = usb_control_msg( usbhandle,
		       USB_ENDPOINT_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		       req, val, index, (char*) buf, buflen, timeout );
  if	( n != buflen )
    {
      fprintf( stderr, "USB write error: expected %d, got %d\n", buflen, n );
      return -1;
    }
  return 0;
}

static int usb_in ( int req, int val, int index, char *buf, int buflen, int umax )
{
  int	n;
  int	timeout;
  
  timeout = USB_TIMEOUT + (buflen * umax) / 1000;
  n = usb_control_msg( usbhandle,
		       USB_ENDPOINT_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
		       req, val, index, (char*) buf, buflen, timeout );
  if	( n != buflen )
    {
      fprintf( stderr, "USB read error: expected %d, got %d\n", buflen, n );
      return -1;
    }
  return n;
}

static int usbspi_setpin(PROGRAMMER * pgm, int pin, int value) {
  char buffer[8];
  int ret;


  if (value) {
    ret = usb_out(USBSPI_SETPIN, pin, 2, buffer, 0, 50);
    if (ret < 0) {
      fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
      return -1;
    }
  } else {
    ret = usb_out(USBSPI_CLRPIN, pin, 2, buffer, 0, 50);
    if (ret < 0) {
      fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
      return -1;
    }
  }
  if (pgm->ispdelay > 1)
    bitbang_delay(pgm->ispdelay);
  
  return 0;
}


static int usbspi_getpin(PROGRAMMER * pgm, int pin)
{
  char buffer[1];
  int ret;

  ret = usb_in(USBSPI_GETPIN, pin, 0, buffer, 1, 200);
  if (ret < 0) {
    fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
    return -1;
  }

  if (buffer[0] & (1 << pin))
    return 1;
  else
    return 0;
}


static int usbspi_highpulsepin(PROGRAMMER * pgm, int pin) {
  char buffer[8];
  int ret;

  if (usbhandle) {
    ret = usb_out(USBSPI_SETPIN, 4 /* reset */, 2, buffer, 0, 50);
    if (ret < 0) {
      fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
      return -1;
    }
    if (pgm->ispdelay > 1)
      bitbang_delay(pgm->ispdelay);

    ret = usb_out(USBSPI_CLRPIN, 4 /* reset */, 2, buffer, 0, 50);
    if (ret < 0) {
      fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
      return -1;
    }
    if (pgm->ispdelay > 1)
      bitbang_delay(pgm->ispdelay);
  }

  return 0;
}

static void usbspi_powerup(PROGRAMMER * pgm)
{
  int ret, sck_period=10;
  char buffer[8];

  if (usbhandle) {
    ret = usb_out(USBSPI_POWERUP, sck_period, 2, buffer, 0, 50);
    if (ret < 0) {
      fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
    }
  }
  usleep(100000);
}

/*
 * transmit an AVR device command and return the results; 'cmd' and
 * 'res' must point to at least a 4 byte data buffer
 */
int usbspi_cmd(PROGRAMMER * pgm, unsigned char cmd[4],
                   unsigned char res[4])
{
  int i, ret;
  unsigned char buffer[4];
 
  /* 
     //one byte at a time
   for (i=0; i<4; i++) {
    ret = usb_in(USBSPI_SPI, cmd[i], 0, res+i, 1, 200);
    if (ret < 0) {
      fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
      return -1;
    }

    if (ret != 1) {
      fprintf(stderr, "Bad response\n");
      return -1;
    }
  }
  */

  ret = usb_out(USBSPI_WRITESPI4, 0, 0, cmd, 4, 200);
  if (ret < 0) {
    fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
    return -1;
  }
  ret = usb_in(USBSPI_READSPI4, 0, 0, res, 4, 200);
  if (ret < 0) {
    fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
    return -1;
  }

  if(verbose >= 2)
    {
      fprintf(stderr, "usbspi_cmd(): [ ");
      for(i = 0; i < 4; i++)
	fprintf(stderr, "%02X ", cmd[i]);
      fprintf(stderr, "] [ ");
      for(i = 0; i < 4; i++)
	{
	  fprintf(stderr, "%02X ", res[i]);
	}
      fprintf(stderr, "]\n");
    }
  
  return 0;
}

static void usbspi_disable(PROGRAMMER * pgm)
{
  /* Do nothing */

  return;
}

static void usbspi_enable(PROGRAMMER * pgm)
{
  /* Do nothing */

  return;
}

static int usbspi_open(PROGRAMMER * pgm, char * port)
{
  struct usb_bus *bus;
  struct usb_device *dev;
  int dev_found;

  usb_init();

  usb_find_busses();
  usb_find_devices();
  
  usbhandle = NULL;
  dev_found = FALSE;
  for (bus = usb_get_busses(); bus && !dev_found; bus = bus->next) {
    for (dev = bus->devices; dev && !dev_found; dev = dev->next) {
      if ((dev->descriptor.idVendor == USBSPI_VID) && 
	  (dev->descriptor.idProduct == USBSPI_PID)) {
	dev_found = TRUE;
	usbhandle = usb_open(dev);
	fprintf(stderr, "Found USB-SPI device with matching VID/PID!\n");
      }
    }
  }
   
  if (!dev_found) {
      /* Couldn't find the VID/PID pair */
      fprintf(stderr,
	      "%s: error: could not find USB device "
	      "\"USBSPI\" with vid=0x%x pid=0x%x\n",
  	      progname, USBSPI_VID, USBSPI_PID);
      exit(1);
  }

  return 0;
}


static void usbspi_close(PROGRAMMER * pgm)
{
  int ret;
  char buffer[8];

  if (usbhandle) {
    ret = usb_out(USBSPI_POWERDOWN, 0, 2, buffer, 0, 50);
    if (ret < 0) {
      fprintf(stderr, "Unable to send vendor request, ret = %d...\n", ret);
    }
    
    usb_close(usbhandle);
  }
}

static void usbspi_display(PROGRAMMER * pgm, char * p)
{
  // get version?

  fprintf(stderr, "USBSPI programmer - www.adafruit.com/make/usbspi\n");
}

void usbspi_initpgm(PROGRAMMER * pgm)
{
  strcpy(pgm->type, "USBSPI");

  //pgm->rdy_led        = usbspi_rdy_led;
  //pgm->err_led        = usbspi_err_led;
  //pgm->pgm_led        = usbspi_pgm_led;
  //pgm->vfy_led        = usbspi_vfy_led;
  pgm->initialize     = bitbang_initialize;
  pgm->display        = usbspi_display;
  pgm->enable         = usbspi_enable;
  pgm->disable        = usbspi_disable;
  pgm->powerup        = usbspi_powerup;
  //  pgm->powerdown      = usbspi_powerdown;
  pgm->program_enable = bitbang_program_enable;
  pgm->chip_erase     = bitbang_chip_erase;
  pgm->cmd            = usbspi_cmd;
  pgm->open           = usbspi_open;
  pgm->close          = usbspi_close;
  pgm->setpin         = usbspi_setpin;
  pgm->getpin         = usbspi_getpin;
  pgm->read_byte      = avr_read_byte_default;
  pgm->write_byte     = avr_write_byte_default;
  pgm->highpulsepin   = usbspi_highpulsepin;

  pgm->pinno[PIN_AVR_SCK] = 7;
  pgm->pinno[PIN_AVR_MOSI] = 6;
  pgm->pinno[PIN_AVR_MISO] = 5;
  pgm->pinno[PIN_AVR_RESET] = 4;
  pgm->pinno[PIN_LED_PGM] = 0;
}

#else  /* !HAVE_LIBUSB */

void usbspi_initpgm(PROGRAMMER * pgm)
{
  fprintf(stderr,
	  "%s: libusb access not available in this configuration\n",
	  progname);
}

#endif /* HAVE_LIBUSB */
